import sys, time
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileProgram, compileShader

# ---------- Globals ----------
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0

# ---------- Lattice params ----------
lattice_width = 32
num_instances_base = 500_000
sub_tile_height = 256
max_tex_height = 2048
threshold = np.sqrt(1.6180339887)
phi = 1.6180339887
phi_powers = np.array([1.0 / pow(phi, 7*(i+1)) for i in range(72)], dtype=np.float32)

tile_heights = []

# ---------- Vertex Shader ----------
VERTEX_SRC = """
#version 330
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){
    texCoord = (pos + 1.0)*0.5;
    gl_Position = vec4(pos,0,1);
}
"""

# ---------- Fragment Shader with recursion depth visualization ----------
FRAGMENT_SRC = """
#version 330
in vec2 texCoord;
out vec4 fragColor;

uniform sampler2D latticeTex;
uniform float cycle;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeHeight;
uniform int yOffset;

// Recursive analog slot computation
float Dn_r(int n, float r_dim, float omega){
    float F_n = float(n <= 2 ? 1 : 1 + n/2.0); // placeholder Fibonacci
    float P_n = float(n <= 7 ? n*0.618 : 1.0); // placeholder primes
    float wave = mod(float(n),3.0) == 0.0 ? 0.3 : (mod(float(n),3.0) == 1.0 ? 0.0 : -0.3);
    float base = sqrt(1.618 * F_n * pow(2.0,float(n)) * P_n * omega) * pow(r_dim,float(n));
    return base + wave;
}

// Hybrid slot: sum of Dn_r contributions
float hybrid_slot(float val, float r_dim, float omega){
    float accum = val;
    for(int n=1; n<=8; n++){
        accum += Dn_r(n, r_dim, omega);
    }
    return accum;
}

void main(){
    int x = int(texCoord.x * 32.0);
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    float val = texelFetch(latticeTex, ivec2(x,y - yOffset), 0).r;
    float r_dim = float(y) / float(latticeHeight); // recursion depth
    float slot_val = hybrid_slot(val, r_dim, sin(omegaTime));

    // Map recursion depth to color gradient
    float red = r_dim;                      // left-brain analog
    float green = abs(sin(slot_val));       // hybrid modulation
    float blue = 1.0 - r_dim;               // right-brain analog
    fragColor = vec4(red, green, blue, 1.0);
}
"""

# ---------- GL Initialization ----------
def init_gl():
    global shader, vao, textures, fbos, tile_heights
    shader = compileProgram(compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
                            compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER))
    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1], dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER, vbo)
    glBufferData(GL_ARRAY_BUFFER, verts.nbytes, verts, GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)
    reinit_lattice(num_instances_base)

# ---------- Lattice Reinit ----------
def reinit_lattice(new_num_instances):
    global textures, fbos, tile_heights
    for tex_pair in textures:
        glDeleteTextures(tex_pair)
    for fbo_pair in fbos:
        glDeleteFramebuffers(2, fbo_pair)
    textures.clear()
    fbos.clear()
    tile_count = (new_num_instances + max_tex_height - 1) // max_tex_height
    tile_heights[:] = [min(max_tex_height, new_num_instances - i*max_tex_height) for i in range(tile_count)]
    for th in tile_heights:
        tex_pair = glGenTextures(2)
        fbo_pair = glGenFramebuffers(2)
        for i in range(2):
            glBindTexture(GL_TEXTURE_2D, tex_pair[i])
            glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, lattice_width, th, 0, GL_RGBA, GL_FLOAT, None)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
            glBindFramebuffer(GL_FRAMEBUFFER, fbo_pair[i])
            glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, tex_pair[i],0)
        textures.append(tex_pair)
        fbos.append(fbo_pair)
        init_chunk = np.zeros((th,lattice_width,4),dtype=np.float32)
        glBindTexture(GL_TEXTURE_2D, tex_pair[0])
        glTexSubImage2D(GL_TEXTURE_2D,0,0,0,lattice_width,th,GL_RGBA,GL_FLOAT,init_chunk)
    glBindFramebuffer(GL_FRAMEBUFFER,0)

# ---------- Display ----------
def display():
    global cycle, omega_time, current
    next_idx = 1-current
    for t, th in enumerate(tile_heights):
        glBindFramebuffer(GL_FRAMEBUFFER, fbos[t][next_idx])
        glViewport(0,0,lattice_width,th)
        glUseProgram(shader)
        glUniform1i(glGetUniformLocation(shader,"latticeTex"),0)
        glUniform1f(glGetUniformLocation(shader,"cycle"),cycle)
        glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
        glUniform1fv(glGetUniformLocation(shader,"phiPowers"),72,phi_powers)
        glUniform1f(glGetUniformLocation(shader,"threshold"),threshold)
        glUniform1i(glGetUniformLocation(shader,"latticeHeight"), th)
        glUniform1i(glGetUniformLocation(shader,"yOffset"), 0)
        glActiveTexture(GL_TEXTURE0)
        glBindTexture(GL_TEXTURE_2D, textures[t][current])
        glBindVertexArray(vao)
        glDrawArrays(GL_TRIANGLES,0,6)

    glBindFramebuffer(GL_FRAMEBUFFER,0)
    glViewport(0,0,1280,720)
    for t, th in enumerate(tile_heights):
        glBindTexture(GL_TEXTURE_2D, textures[t][next_idx])
        glDrawArrays(GL_TRIANGLES,0,6)

    glutSwapBuffers()
    cycle += 1
    omega_time += 0.05
    current = next_idx

# ---------- Idle ----------
def idle():
    glutPostRedisplay()

# ---------- Main ----------
def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL Left/Right Brain Recursion Visualization")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
